module net.BurtonRadons.parse.scope;

/** Classes for name scoping. */

/** Base scope. */
class Scope
{
    import net.BurtonRadons.parse.declaration;
    import net.BurtonRadons.parse.error;
    import net.BurtonRadons.parse.parser;
    import net.BurtonRadons.parse.statement;

    Parser parser;
    Scope parent; /**< Parent scope or null. */
    Statement [] list; /**< Statements inside of the scope. */
    Symbol [char []] decls; /**< Declarations, setup in a semantic pass. */
    Declaration root; /**< Root declaration this scope is from or null. */

    this (Parser parser)
    {
        this.parser = parser;
    }

    void add (Statement parent, Statement item)
    {
        item.parent = parent;
        list ~= item;
    }

    void define (Symbol symbol)
    {
        if (symbol.name === null)
            return;
        decls [symbol.name] = symbol;
    }

    void semanticError (MarkedError error)
    {
        parser.semanticError (error);
    }

    void semantic0 ()
    {
        for (int c; c < list.length; c ++)
            list [c] = list [c].semantic0 (this);
    }

    void semantic1 ()
    {
        for (int c; c < list.length; c ++)
            list [c] = list [c].semantic1 (this);
    }

    void semantic2 ()
    {
        for (int c; c < list.length; c ++)
            list [c] = list [c].semantic2 (this);
    }

    /** Try to find the named symbol in this or a parent scope. */
    Symbol find (char [] name)
    {
        if (name in decls)
            return decls [name];
        if (parent !== null)
            return parent.find (name);
        return null;
    }

    /** Find the root declaration this scope is in or null. */
    Declaration findRoot ()
    {
        Scope d = this;

        while (d !== null && d.root === null)
            d = d.parent;
        if (d === null)
            return null;
        return d.root;
    }
}

/** Module. */
class Module : Scope
{
    char [] filename; /**< Source filename of this module. */

    /** Assign fields and then read. */
    this (Parser parser)
    {
        super (parser);
        if (parser.source !== null)
            this.filename = parser.source.mark.filename;
    }

    /** Run the semantic pass on the module. */
    void semantic ()
    {
        semantic0 ();
        semantic1 ();
        semantic2 ();
    }
}

/** A symbol name; a period-separated set of identifiers. */
class Symbol
{
    import net.BurtonRadons.parse.declaration;
    import net.BurtonRadons.parse.expression;
    import net.BurtonRadons.parse.type;
    
    Type type; /**< Symbol type or null. */
    Expression init; /**< Initial value or null. */
    char [] name; /**< Symbol identifier. */
    Declaration decl; /**< Declaration this is attached to. */

    bit isin = true; /**< If set, this is an "in" parameter. */
    bit isout; /**< If set, this is an "out" parameter. */
    bit isinout; /**< If set, this is an "inout" parameter. */

    this ()
    {
    }

    this (char [] name)
    {
        this.name = name;
    }

    this (Type type)
    {
        this.type = type;
    }

    /** Convert to std.string. */
    char [] toString ()
    {
        return name;
    }

    Symbol semantic0 (Scope scope)
    {
        if (type !== null)
            type = type.semantic0 (scope);
        if (init !== null)
            init = init.semantic0 (scope);
        return this;
    }

    Symbol semantic1 (Scope scope)
    {
        if (type !== null)
            type = type.semantic1 (scope);
        if (init !== null)
            init = init.semantic1 (scope);
        return this;
    }

    Symbol semantic2 (Scope scope)
    {
        if (type !== null)
            type = type.semantic2 (scope);
        if (init !== null)
            init = init.semantic2 (scope);
        return this;
    }
}
